unit Ltnumedt;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Menus, DsgnIntf;

type
  TLTNumEdit = class(TCustomEdit)
  private
    FAbout: string;
    FMirror: string;
    FMaxLength: integer;
    FMaxDecimals: integer;
    FNegativeAllowed: boolean;
    FTabOnEnter: boolean;
    FAlignment: TAlignment;
    FFloat: boolean;
    procedure SetMaxLength(value: integer);
    procedure SetMaxDecimals(value: integer);
    procedure SetText(value: string);
    procedure SetFloat(value: boolean);
    procedure SetEditRect;
    function GetFormatText: string;
    procedure WMCut(var Message: TMessage); message WM_CUT;
    procedure WMPaste(var Message: TMessage); message WM_PASTE;
  protected
    procedure CreateWnd; override;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure DoEnter; override;
    function IsValid(s: string): boolean;
    procedure GetSel(var SelStart: Integer; var SelStop: Integer);
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyUp(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: char); override;
  public
    constructor Create(AOwner: TComponent);
    procedure ShowAbout;
  published
    property About: string read FAbout write FAbout stored false;
    property Alignment: TAlignment read FAlignment write FAlignment;
    property AutoSelect;
    property AutoSize;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Float: boolean read FFloat write SetFloat;
    property Font;
    property HideSelection;
    property MaxDecimals: integer read FMaxDecimals write SetMaxDecimals;
    property MaxLength: integer read FMaxLength write SetMaxLength;
    property NegativeAllowed: boolean read FNegativeAllowed write FNegativeAllowed;
    property OEMConvert;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOnEnter: boolean read FTabOnEnter write FTabOnEnter;
    property TabOrder;
    property TabStop;
    property Text: string read FMirror write SetText;
    property FormatText: string read GetFormatText;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

uses ClipBrd;

function HasComma(s: string): boolean;
var
  i: integer;
begin
  result:=false;
  if s='' then exit;
  for i:=1 to length(s) do if s[i]='.' then
  begin
    result:=true;
    exit;
  end;
end;

function LeftString(s: string; i: integer): string;
begin
  result:='';
  if (i=0)or(s='') then exit;
  result:=copy(s,1,i);
end;

function RightStringFrom(s: string; i: integer): string;
begin
  result:='';
  if (s='')or(i>length(s)) then exit;
  result:=copy(s,i,length(s)-i+1);
end;

constructor TLTNumEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FMirror:='';
  {MaxLength:=15;
  MaxDecimals:=0;
  NegativeAllowed:=true;
  FTabOnEnter:=false;
  FAlignment:=taLeftJustify;
  FFloat:=true;}
  inherited Text:='';
end;

procedure TLTNumEdit.CreateWnd;
begin
  inherited CreateWnd;
  SetEditRect;
end;

procedure TLTNumEdit.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  case Alignment of
    taLeftJustify  : Params.Style:=Params.Style or ES_LEFT or ES_MULTILINE;
    taRightJustify : Params.Style:=Params.Style or ES_RIGHT or ES_MULTILINE;
    taCenter       : Params.Style:=Params.Style or ES_CENTER or ES_MULTILINE;
  end;
end;

procedure TLTNumEdit.DoEnter;
begin
  (*Text := FEditText; {change the displayed text to the EditText}*)
  inherited DoEnter;
  if AutoSelect then SelectAll;
end;

function TLTNumEdit.IsValid(s: string): boolean;
var
  i,L,n,d: integer;
  fcomma: boolean;
  MaxBefore: integer;
begin
  result:=true;
  if s='' then exit;
  result:=false;
  if (not NegativeAllowed)and(s[1]='-') then exit;
  fcomma:=false;
  L:=length(s);
  MaxBefore:=MaxLength-FMaxDecimals;
  if FMaxDecimals<>0 then dec(MaxBefore);
  i:=1;
  n:=0;
  d:=0;
  if s[i]='-' then inc(i);
  while i<=L do
  begin
    if i>MaxLength then exit;
    if not(s[i] in ['0'..'9','.']) then exit;
    if (s[i]='.') then
    begin
      if fcomma then exit;
      fcomma:=true;
    end
    else
    begin
      if fcomma then inc(n) else inc(d);
      if (not FFloat)
      and((d>MaxBefore)or(n>FMaxDecimals)) then exit;
    end;
    inc(i);
  end;
  result:=true;
end;

procedure TLTNumEdit.SetMaxLength(value: integer);
begin
  inherited MaxLength:=value;
  FMaxLength:=value;
  Text:='';
end;

procedure TLTNumEdit.SetMaxDecimals(value: integer);
begin
  FMaxDecimals:=value;
  Text:='';
end;

procedure TLTNumEdit.SetText(value: string);
var
  s: string;
  i,j: integer;
  L: integer;
begin
  s:='';
  if value<>'' then
  begin
    L:=length(value);
    i:=1;while (i<=L)and(value[i]=' ') do inc(i);
    j:=i;while (j<=L)and(value[i]<>' ')do inc(j);
    s:=copy(value,i,j-i+1);
  end;
  if not IsValid(s) then exit;
  inherited Text:=s;
  FMirror:=s;
end;

function TLTNumEdit.GetFormatText: string;
var
  s: string;
  i,j: integer;
begin
  s:=FMirror;
  if s='' then s:='0';
  if (FFloat)or(FMaxDecimals=0) then
  begin
    if s[length(s)]='.' then dec(byte(s[0]));
    if s='' then s:='0';
    while length(s)<MaxLength do s:=' '+s;
    result:=s;
  end
  else
  begin
    i:=1;while(i<=length(s))and(s[i]<>'.') do inc(i);
    if i>length(s) then s:=s+'.';
    i:=FMaxDecimals-(length(s)-i);
    while i>0 do
    begin
      s:=s+'0';
      dec(i);
    end;
    while length(s)<FMaxLength do s:=' '+s;
    result:=s;
  end;
end;

procedure TLTNumEdit.SetFloat(value: boolean);
var
  OldFloat: boolean;
begin
  OldFloat:=FFloat;
  if value=OldFloat then exit;
  FFloat:=value;
  if IsValid(FMirror) then exit;
  FFloat:=OldFloat;
end;

procedure TLTNumEdit.GetSel(var SelStart,SelStop: integer);
var
  LI: LongInt;
begin
  LI:=SendMessage(Handle,EM_GETSEL,0,0);
  SelStart:=LoWord(LI);
  SelStop:=HiWord(LI);
end;

procedure TLTNumEdit.KeyDown(var Key: Word; Shift: TShiftState);
var
  SelStart,SelStop: integer;
begin
  inherited KeyDown(Key,Shift);
  GetSel(SelStart,SelStop);
  case Key of
    VK_DELETE:
    begin
      if SelStart=SelStop then
        FMirror:=LeftString(FMirror,SelStart)+RightStringFrom(FMirror,SelStop+2)
      else
        FMirror:=LeftString(FMirror,SelStart)+RightStringFrom(FMirror,SelStop+1);
    end;
    VK_RETURN: if FTabOnEnter then
    begin
      SendMessage(GetParentForm(Self).Handle,WM_NEXTDLGCTL,0,0);
      Key:=0;
    end;
  end;
end;

procedure TLTNumEdit.KeyUp(var Key: Word; Shift: TShiftState);
begin
  inherited KeyUp(Key,Shift);
end;

procedure TLTNumEdit.KeyPress(var Key: char);
var
  SelStart,SelStop: integer;
  Test: string;
begin
  inherited KeyPress(Key);
  GetSel(SelStart,SelStop);
  if Key=#8 then
  begin
    if (FMirror='')
    or ((SelStart=SelStop)and(SelStart=0)) then
    begin
      Key:=#0;
      exit;
    end;
    if SelStart<>SelStop then
      FMirror:=LeftString(FMirror,SelStart)+RightStringFrom(FMirror,SelStop+1)
    else
      FMirror:=LeftString(FMirror,SelStart-1)+RightStringFrom(FMirror,SelStop+1);
    exit;
  end;
  if (not (Key in ['0'..'9','.','-'])) then
  begin
    Key:=#0;
    exit;
  end;
  if (not NegativeAllowed)and(Key='-') then
  begin
    Key:=#0;
    exit;
  end;
  (*if (SelStart<>SelStop) then*)
  begin
    Test:=LeftString(FMirror,SelStart)+Key+RightStringFrom(FMirror,SelStop+1);
    if (not IsValid(Test)) then
    begin
      Key:=#0;
      exit;
    end;
    FMirror:=Test;
    exit;
  end;
  (*
  if (Key='.')and((FMaxDecimals=0)or(HasComma(FMirror))) then
  begin
    Key:=#0;
    exit;
  end;
  *)
end;

procedure TLTNumEdit.SetEditRect;
var
  R: TRect;
begin
  SendMessage(Handle,EM_GETRECT,0,LongInt(@R));
  R.Bottom:=ClientHeight+1;  {windows paint bug: +1}
  R.Right:=ClientWidth-2;
  R.Top:=0;
  R.Left:=0;
  SendMessage(Handle,EM_SETRECTNP,0,LongInt(@R));
  SendMessage(Handle,EM_GETRECT, 0,LongInt(@R));
end;

procedure TLTNumEdit.WMCut(var Message: TMessage);
var
  SelStart,SelStop: integer;
begin
  GetSel(SelStart,SelStop);
  if SelStart=SelStop then exit;
  CopyToClipboard;
  FMirror:=LeftString(FMirror,SelStart)+RightStringFrom(FMirror,SelStop+1);
  inherited;
end;

procedure TLTNumEdit.WMPaste(var Message: TMessage);
var
  ins: string;
  Test: string;
  SelStart,SelStop: integer;
begin
  GetSel(SelStart,SelStop);
  Clipboard.Open;
  ins:=Clipboard.AsText;
  Clipboard.Close;
  Test:=LeftString(FMirror,SelStart)+Ins+RightStringFrom(FMirror,SelStop+1);
  if not IsValid(Test) then exit;
  FMirror:=Test;
  inherited;
end;

procedure TLTNumEdit.ShowAbout;
var
  msg: string;
const
  carriage_return = chr(13);
  copyright_symbol = chr(169);
begin
  msg:='TLTNumEdit version 1.0';
  AppendStr(msg,carriage_return);
  AppendStr(msg,'A LunchTime Development');
  AppendStr(msg,carriage_return);
  AppendStr(msg,'(16 bit version)');
  AppendStr(msg,carriage_return);
  AppendStr(msg,carriage_return);
  AppendStr(msg,'Copyright ');
  AppendStr(msg,copyright_symbol);
  AppendStr(msg,' 1997 Danny Smalle');
  AppendStr(msg,carriage_return);
  ShowMessage(msg);
end;

{-------------------------------------------------------------}
type
  TAboutProperty = class(TPropertyEditor)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
    function GetValue:string; override;
  end;

procedure TAboutProperty.Edit;
{call the 'About' dialog window when clicking on ... in the Object Inspector}
begin
  TLTNumEdit(GetComponent(0)).ShowAbout;
end;

function TAboutProperty.GetAttributes: TPropertyAttributes;
{set up to display a string in the Object Inspector}
begin
  GetAttributes := [paDialog, paReadOnly];
end;

function TAboutProperty.GetValue: String;
{set string to appear in the Object Inspector}
begin
  GetValue := '(About)';
end;


procedure Register;
begin
  RegisterComponents('LunchTime', [TLTNumEdit]);
  RegisterPropertyEditor(TypeInfo(String),TLTNumEdit,'About',TAboutProperty);
end;

end.

